//
//  $Id: WXKPasswordWindowController.m 108 2009-06-24 14:54:10Z fujidana $
//  Copyright 2006 FUJIDANA. All rights reserved.
//
//
//  Some part of this source code is derived from PasswordController.m, 
//  written by raktajino in AH-K3001V Bookmark Utility and distributed
//  under BSD license.
//
//
//  PasswordController.m
//  FileUtility
//
//  Created by raktajino on Thu 2005/03/07.
//  Copyright (c) 2005 raktajino. All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
// IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
// INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
// NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
// THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


#import "WXKPasswordWindowController.h"
#import <Security/Security.h> 


const char *kServiceName = "AH-K3001V Password"; // service name in Keychain
const char *kAccountName = "AH-K3001V";          // account name in Keychain


@interface WXKPasswordWindowController ()

- (NSString *)passwordInKeyChains;
- (void)storePassword;
- (int)runModal;
//- (void)beginSheetModalForWindow:(NSWindow *)window modalDelegate:(id)delegate didEndSelector:(SEL)didEndSelector contextInfo:(void *)contextInfo;


@end


@implementation WXKPasswordWindowController

#pragma mark Class method

+ (id)sharedWindowController
{
	static WXKPasswordWindowController *staticPasswordWindowController;
	if (!staticPasswordWindowController)
	{
		staticPasswordWindowController = [[WXKPasswordWindowController alloc] initWithWindowNibName:@"PasswordWindow"];
	}
	return staticPasswordWindowController;
}

#pragma mark Deallocation

- (void)dealloc
{
	[_passwordInMemory release];
	[super dealloc];
}

#pragma mark Accessor methods

- (BOOL)requestsUserInput
{
	return _requestsUserInput;
}

- (void)setRequestsUserInput:(BOOL)flag
{
	_requestsUserInput = flag;
}

- (NSString *)password
{
	NSString *password = nil;
	
	// Seek a password in the keychains (in case it is requested to force user input, skip this step)
	if ([self requestsUserInput] == NO)
	{
		password = (_passwordInMemory) ? _passwordInMemory : [self passwordInKeyChains];
	}
	
	// In case a password is not found on memory or keychains, 
	// or it is requested to force user inputing password,
	if (password == nil)
	{
		if ([self runModal] == NSOKButton)
		{
			[self setRequestsUserInput:NO];
			password = (_passwordInMemory) ? _passwordInMemory : [self passwordInKeyChains];
		}
	}
	return password;
}

#pragma mark Action methods

/* 
   invoked when user clicks the OK button in the sheet.
*/
- (IBAction)ok:(id)sender
{
	[self storePassword];
	
	if ([[self window] isSheet])
	{
		[NSApp endSheet:[self window] returnCode:NSOKButton];
	}
	else if ([self window] == [NSApp modalWindow])
	{
		[NSApp stopModalWithCode:NSOKButton];
	}
	[[self window] orderOut:nil];
}

/* 
   invoked when user clicks the cancel button in the sheet.
*/
- (IBAction)cancel:(id)sender
{
	if ([[self window] isSheet])
	{
		[NSApp endSheet:[self window] returnCode:NSCancelButton];
	}
	else if ([self window] == [NSApp modalWindow])
	{
		[NSApp stopModalWithCode:NSCancelButton];
	}
	[[self window] orderOut:nil];
}

#pragma mark Displaying dialog

/* 
  Display application modal dialog. Returned value is NSAlertFirstButtonReturn or NSAlertSecondButtonReturn.
*/
- (int)runModal
{
	return [NSApp runModalForWindow:[self window]];
}

- (void)beginSheetModalForWindow:(NSWindow *)window modalDelegate:(id)delegate didEndSelector:(SEL)didEndSelector contextInfo:(void *)contextInfo
{
	[NSApp beginSheet:[self window] modalForWindow:window modalDelegate:delegate didEndSelector:didEndSelector contextInfo:contextInfo];
}

#pragma mark Private methods

/* 
   Return a password stored in Keychain. If not found, return nil. 
*/
- (NSString *)passwordInKeyChains
{
	NSString *password = nil;
	
	OSStatus result;
	UInt32 passwordLength = 0;
	void   *passwordData  = nil;
	result = SecKeychainFindGenericPassword(NULL,
											strlen(kServiceName), kServiceName,
											strlen(kAccountName), kAccountName,
											&passwordLength, &passwordData, NULL);
	if(result == noErr)
	{
		// convert obtained password into NSString
		password = [[[NSString alloc] initWithBytes:passwordData length:passwordLength encoding:NSUTF8StringEncoding] autorelease];
		SecKeychainItemFreeContent(NULL, passwordData);
	}
	return password;
}

/* 
   Save a password into application memory or Keychain.
*/
- (void)storePassword
{
	// save a password into Keychain.
	if ([saveInKeychainButton state] == NSOnState)
	{
		// convert the password into UTF8 encoding.
		NSData *newPassword = [[passwordField stringValue] dataUsingEncoding:NSUTF8StringEncoding];
		
		// check whether password is already registered in Keychain or not.
		OSStatus           result;
		UInt32             passwordLength = 0;
		void              *passwordData   = nil;
		SecKeychainItemRef itemRef        = nil;
		result = SecKeychainFindGenericPassword(NULL,
												strlen(kServiceName), kServiceName,
												strlen(kAccountName), kAccountName,
												&passwordLength, &passwordData, &itemRef);
		
		if (result == noErr)
		{
			// Since another password has already been registered in Keychain, replace it with newer value.
			SecKeychainItemFreeContent(NULL, passwordData);
			SecKeychainAttribute attrs[] = {
			{ kSecAccountItemAttr, strlen(kAccountName), (char *)kAccountName },
			{ kSecServiceItemAttr, strlen(kServiceName), (char *)kServiceName }
			};
			const SecKeychainAttributeList attributes = { sizeof(attrs) / sizeof(attrs[0]), attrs };
			SecKeychainItemModifyAttributesAndData(itemRef, &attributes, [newPassword length], [newPassword bytes]);
			CFRelease(itemRef);
		}
		
		if (result == errSecItemNotFound)
		{
			// since password has not been registered in Keychain, make new Keychain item.
			SecKeychainAddGenericPassword(NULL,
										  strlen(kServiceName), kServiceName,
										  strlen(kAccountName), kAccountName,
										  [newPassword length], [newPassword bytes], NULL);
		}
		
		// clear the password on memory, if it exists.
		[_passwordInMemory release];
		_passwordInMemory = nil;
	}
	// save a password onto the application memory, not in Keychain.
	else
	{
		[_passwordInMemory release];
		_passwordInMemory = [[passwordField stringValue] copy]; 
	}
}

@end